#!/usr/bin/env python
# encoding: utf-8
"""
PlotSection.py

Plotting routine to do vertical sections
"""

import numpy
import datetime
import netCDF4
import matplotlib
import pylab
import os
import subprocess
import csv
from mpl_toolkits.basemap import Basemap


def prePlot():
    pylab.figure(figsize=(10, 6))


def postPlot():
    pylab.xlabel('Distance [km]', fontsize=14)
    pylab.ylabel('Height [km]', fontsize=14)
    pylab.ylim([0.,7.])
    pylab.xticks(axis_locs,dist_ticks)
    pylab.yticks(numpy.arange(0.,8.,1.))
    pylab.grid()
    pylab.figtext( .780, .055, u'ms$^{-1}$', fontsize=14)
    pylab.figtext( .725, .055, u'Jkg$^{-1}$', fontsize=14)
    pylab.savefig(pngfile, dpi=600)
    pylab.clf()
    pylab.close()
    id1 = subprocess.Popen(['convert', '-trim', pngfile, pngfile])
    id1.wait()


def plotMap():
    pylab.figure(figsize=(10, 6))
    shapefiles_path=os.path.join('/', 'opt', 'python', 'Shapefiles', 
                                 'Transformed')
    shapefile_joklarname = 'joklar'
    shapefile_joklar = os.path.join(shapefiles_path, shapefile_joklarname)
    m=Basemap(projection='stere', lat_ts=dataset.TRUELAT1,
              lat_0=dataset.MOAD_CEN_LAT,lon_0=dataset.STAND_LON,
              llcrnrlon=-18.45, llcrnrlat=63.58, 
              urcrnrlon=-14.6, urcrnrlat=64.82,
              resolution='i')
    m.readshapefile(shapefile_joklar, shapefile_joklarname,
                    linewidth=1., color='black')
    m.drawcoastlines(linewidth=0.5)
    m.drawparallels(numpy.arange(63.,67.,.5), labels=[1,0,0,0])
    m.drawmeridians(numpy.arange(-26.,-12.,1.), labels=[0,0,0,1])
    x,y = m( xlon,xlat)
    m.contour(x,y,hgt,numpy.arange(200,2240,200),colors='gray',linewidths=0.5)
    m.drawmapscale(-18.1, 63.7, -19., 65., 20., barstyle='fancy', fontsize=12)
    x,y = m( [ xlon[j0,i0],xlon[j1,i1] ], [ xlat[j0,i0],xlat[j1,i1] ] )
    m.plot(x, y, 'r-', linewidth=2.)
    pylab.savefig('map'+pngfile)
    pylab.clf()
    pylab.close()
    id1 = subprocess.Popen(['convert', '-trim', 'map'+pngfile, 'map'+pngfile])
    id1.wait()


def interpSection(i0,j0,i1,j1,Var3d):
    # Interpolate for every layer
    i = 0
    for Var2d in Var3d:
        Varline = interpLine(i0,j0,i1,j1,Var2d)
        Varline.shape = 1,Varline.size
        if i == 0:
            i = 1
            Varsection = Varline
        else:
            Varsection = numpy.append(Varsection,Varline,axis=0)
    return Varsection


def interpLine(i0,j0,i1,j1,Var2d):
    # Interpolate for each layer
    lengthout = 2.0 * numpy.amax([i1-i0,j1-j0])
    xout = numpy.linspace(i0,i1,lengthout)
    yout = numpy.linspace(j0,j1,lengthout)
    xout.shape = 1,lengthout
    yout.shape = 1,lengthout
    xin, yin = numpy.mgrid[0:Var2d.shape[0],0:Var2d.shape[1]]
    Var2d.shape = Var2d.size
    xin.shape = Var2d.size
    yin.shape = Var2d.size
    return matplotlib.mlab.griddata(xin,yin,Var2d,xout,yout)[0,:]


def colors():
    colors = [
        (0.0, 0.0, 0.71390374331550799),
        (0.0, 0.096078431372549025, 1.0),
        (0.0, 0.50392156862745097, 1.0),
        (0.0, 0.89607843137254906, 0.97090449082858954),
        (0.490196078, 1.0, 0.478431373),
        (0.843137255, 1.0, 0.121568627),
        (1.0, 0.698039216, 0.0),
        (1.0, 0.21859114015976777, 0.0),
        (0.71390374331550821, 0.0, 0.0),
        (0.501960784, 0.0, 0.0),
        (0.501960784, 0.0, 0.0),
        ]
    colors = list(colors)
    for index, color in enumerate(colors):
        colors[index] = tuple(color)
    return colors


# Read variables
print 'Reading data'
lat0, lon0  = [ 64.01, -17.1 ]
lat1, lon1  = [ 63.98, -15.96 ]
wrffile, res = 'Data/Okyrrd_ECpl-51sigma_931km/wrfout_d03_2008-11-18_06:00:00', 1
#wrffile, res = 'Data/Okyrrd_ECpl-51sigma_93km/wrfout_d02_2008-11-18_06:00:00', 3
#wrffile, res = 'Data/Okyrrd_ECpl-51sigma_9km/wrfout_d01_2008-11-18_00:00:00', 9
pngfile, t0 = 'sim_section1km_16UTC18nov2008.png', 10
#pngfile, t0 = 'sim_section1km_17UTC18nov2008.png', 11
#pngfile, t0 = 'sim_section1km_19UTC18nov2008.png', 13
#pngfile, t0 = 'sim_section3km_16UTC18nov2008.png', 10
#pngfile, t0 = 'sim_section9km_16UTC18nov2008.png', 16
dataset = netCDF4.Dataset(wrffile)
print dataset.variables['Times'][t0].tostring()
u_stag = dataset.variables['U'][t0]
v_stag = dataset.variables['V'][t0]
w_stag  = dataset.variables['W'][t0]
th  = dataset.variables['T'][t0]
ph  = dataset.variables['PH'][t0]
phb = dataset.variables['PHB'][t0]
tke = 2. * dataset.variables['TKE_PBL'][t0]
hgt = dataset.variables['HGT'][0]
xlat = dataset.variables['XLAT'][0]
xlon = dataset.variables['XLONG'][0]
sinalpha = dataset.variables['SINALPHA'][0]
cosalpha = dataset.variables['COSALPHA'][0]

# Find our location
print 'Finding location'
latlon = xlat + xlon*1j
lat0lon0 = lat0 + lon0*1j
lat1lon1 = lat1 + lon1*1j
j0,i0 = (numpy.abs(latlon-lat0lon0) ==
          numpy.min(numpy.abs(latlon-lat0lon0))).nonzero()
j1,i1 = (numpy.abs(latlon-lat1lon1) ==
          numpy.min(numpy.abs(latlon-lat1lon1))).nonzero()
print ' Startpoint (true): ',  lat0, lon0
print ' Endpoint (true): ', lat1, lon1
print ' Startpoint (model): ', xlat[j0,i0], xlon[j0,i0], [j0,i0]
print ' Endpoint (model): ', xlat[j1,i1], xlon[j1,i1], [j1,i1]

# Calculate variables
print "Work variables!"
z_stag = ( (ph+phb) / 9.81 ) / 1000.
z = 0.5 * (z_stag[0:-1,:,:] +  z_stag[1:,:,:] )
w = 0.5 * ( w_stag[0:-1] + w_stag[1:] )
t = th + 300.
u_destag = 0.5 * ( u_stag[:,:,0:-1] + u_stag[:,:,1:] )
v_destag = 0.5 * ( v_stag[:,0:-1,:] + v_stag[:,1:,:] )
u =  cosalpha*u_destag + sinalpha*v_destag
v = -sinalpha*u_destag + cosalpha*v_destag
ws = (u**2+v**2)**0.5
dir = numpy.mod(270. - (numpy.arctan2(v,u)*180./numpy.pi ), 360.)

# Cross section interpolation/calculation
print 'Work section'
ws_sec = interpSection(j0,i0,j1,i1,ws.squeeze())
u_sec = interpSection(j0,i0,j1,i1,u.squeeze())
v_sec = interpSection(j0,i0,j1,i1,v.squeeze())
w_sec = interpSection(j0,i0,j1,i1,w.squeeze())
t_sec  = interpSection(j0,i0,j1,i1,t.squeeze())
tke_sec  = interpSection(j0,i0,j1,i1,tke.squeeze())
z_sec  = interpSection(j0,i0,j1,i1,z.squeeze())
xin, yin = numpy.mgrid[0:z_sec.shape[0],0:z_sec.shape[1]]
dir_sec    = numpy.mod(270. - (numpy.arctan2(v_sec,u_sec)*180./numpy.pi), 360.)
angle_sec =  numpy.mod(270. - (numpy.arctan2(j1-j0,i1-i0)*180./numpy.pi), 360.)
along_sec = ws_sec * numpy.cos((angle_sec-dir_sec)*numpy.pi/180.)

# Find true distance vector!
dist = numpy.sqrt( ((j1-j0)*res)**2 + ((i1-i0)*res)**2)
dist_ticks = numpy.arange(0,int(dist),10)
axis_locs = numpy.arange(0.,float(len(yin[0,:])),
                          float(len(yin[0,:])) / len(dist_ticks))

# Plot date    
print 'Plotting!'
prePlot()
CT = pylab.contour(yin, z_sec, t_sec, numpy.arange(270,500,2), \
                        linewidth=2, colors='k' )
CFF = pylab.contourf(yin, z_sec, ws_sec, numpy.arange(0,50,5), 
                     extend='max', colors=colors())
cmap = CFF.get_cmap()
cmap.set_over(color=(0.501960784, 0.0, 0.0))
CF = pylab.contour(yin,z_sec,ws_sec,[20,40,60],colors='black', 
                   linestyles='dashed', lineweights=0.5)
pylab.clabel(CT, numpy.arange(270,500,2)[1::2], \
                 inline=1, fmt='%1.0f', fontsize=14)
CFR = pylab.contour(yin,z_sec,along_sec,numpy.arange(-15,1,5),colors='white', 
                   lineweights=2.)
CBF = pylab.colorbar(CFF, extend='max', pad=0.039)
CFTKE = pylab.contourf(yin, z_sec, tke_sec, numpy.arange(5,26,5), 
                       cmap=pylab.cm.gray_r, extend='max')
CBTKE = pylab.colorbar(CFTKE, extend='max',fraction=0.039, pad=0.015)
if res==3:
    ar_int=2
elif res==1:
    ar_int=4
elif res==9:
    ar_int=1
AS = pylab.quiver(  yin[0:-1:2,0:-1:ar_int],
                  z_sec[0:-1:2,0:-1:ar_int],
              along_sec[0:-1:2,0:-1:ar_int],
              10.*w_sec[0:-1:2,0:-1:ar_int],
                  pivot='tip', scale=1600, color='gray')
Z0 = pylab.plot(yin[0,:], z_sec[0,:], 'k-', linewidth=2.)
postPlot()
plotMap()

# Close dataset
dataset.close()
